import { ref } from 'vue';

export default function useDrawing() {
  //画板状态
  const gridSize = { rows: 50, cols: 50 };
  const grid = ref(
    Array.from({ length: gridSize.rows }, () => 
      Array(gridSize.cols).fill('#FFFFFF')
    )
  );
  
  const gridRef = ref(null);
  //颜色设置
  const colors = [
    '#000000', '#FF0000', '#00FF00', '#0000FF', 
    '#FFFF00', '#FF00FF', '#00FFFF', '#FFA500'
  ];
  const selectedColor = ref(colors[0]);

  //计算鼠标位置对应的网格坐标
  const getGridPosition = (event, gridElement) => {
    const rect = gridElement.getBoundingClientRect();
    const x = event.clientX - rect.left;
    const y = event.clientY - rect.top;
  
    const cellSize = Math.min(rect.width / gridSize.cols, rect.height / gridSize.rows);
    const col = Math.floor(x / cellSize);
    const row = Math.floor(y / cellSize);
  
    return { row, col };
  };
  //工具状态
  const isDrawing = ref(false);
  //默认工具为笔刷
  const tool = ref('brush');
  
  //像素颜色设置
  const setPixel = (row, col) => {
    if (!isDrawing.value) return;
    //安全检查
    if (row < 0 || row >= gridSize.rows || col < 0 || col >= gridSize.cols) return;
    
    if (tool.value === 'brush') {
      grid.value[row][col] = selectedColor.value;
    } else if (tool.value === 'eraser') {
      grid.value[row][col] = '#FFFFFF';
    }
  };
  
  //开始绘画
  const startDrawing = (event) => {
    if (!gridRef.value) return;
    
    isDrawing.value = true;
    const { row, col } = getGridPosition(event, gridRef.value);
    setPixel(row, col);
  };
  
  //拖拽中
  const draw = (event) => {
    if (!isDrawing.value || !gridRef.value) return;
    
    const { row, col } = getGridPosition(event, gridRef.value);
    setPixel(row, col);
  };
  
  //结束绘画
  const stopDrawing = () => {
    isDrawing.value = false;
  };
  
  //重置画板
  const resetGrid = () => {
    grid.value = Array.from({ length: gridSize.rows }, () => 
      Array(gridSize.cols).fill('#FFFFFF')
    );
  };
  
  //设置橡皮擦
  const setEraser = () => {
    tool.value = 'eraser';
  };
  
  //设置画笔
  const setBrush = () => {
    tool.value = 'brush';
  };
  //将所有函数和工具接口暴露给外部组件
  return {
    grid,
    gridRef, 
    colors,
    selectedColor,
    tool,
    startDrawing,
    draw, 
    stopDrawing,
    resetGrid,
    setEraser,
    setBrush,
    gridSize
  };
}